package org.lsst.ccs.bootstrap;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.jar.JarFile;
import java.util.jar.Manifest;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import org.apache.commons.cli.BasicParser;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.OptionBuilder;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.freehep.util.VersionComparator;
import org.freehep.util.VersionComparator.Version;
import org.lsst.ccs.bootstrap.resources.BootstrapResourceUtils;
import org.lsst.ccs.bootstrap.resources.ResourcesUtils;
import org.lsst.ccs.bootstrap.util.SystemPropertyMatcher;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.SAXException;

/**
 *
 * Bootstrap main class. Used to launch any CCS application.
 *
 * @author turri
 */
public class Bootstrap {

    // The Options containing the bootstrap command line options.
    private Options bootstrapCommandLineOptions;
    private List<String> additionalCommandLineArguments = new ArrayList<>(), passedAlongOptions = new ArrayList<>();
    // The name of the application to be launched.    
    private static String bootstrapApplication = null;
    static URLClassLoader applicationClassLoader = null;
    private boolean printHelp = false, listApplications = false, showDistributionInfo = false;
    private static boolean verbose = false;
    static final String APPLICATION_OPTION = "application";
    static final String HELP_OPTION = "help";
    static final String VERBOSE_OPTION = "verbose";
    static final String LIST_APPLICATIONS_OPTION = "listApplications";
    private String showProperties = null;
    private boolean showClasspath = false;
    private static final String APPLICATION_ARGS_PROPERTY = "org.lsst.ccs.application.args";
    private static final String APPLICATION_DESCRIPTION_PROPERTY = "org.lsst.ccs.application.description";
    private static Properties bootstrapCmdLineProperties = new Properties();
    private static Map<String, String> additionalClassPathEntriesMap = new HashMap<>();
    private static List<String> additionalClassPathEntriesList = new ArrayList<>();
    private static Properties bootstrapApplicationProperties = null;
    private static final String BOOTSTRAP_ENVIRONMENT_PROPERTY = "org.lsst.ccs.bootstrap.environment";
    private static String distributionMainJar = null;
    static boolean quiet = false;
    private static Class loaderClass = BootstrapUtils.class;

    Bootstrap() {
        this(false);
    }

    Bootstrap(boolean quiet) {
        this.quiet = quiet;
        // define the command line options
        bootstrapCommandLineOptions = new Options();
        // The help option
        bootstrapCommandLineOptions.addOption("h", HELP_OPTION, false, "Print the help message");
        // The verbose option
        bootstrapCommandLineOptions.addOption("v", VERBOSE_OPTION, false, "Turns on verbose statements");
        // The listApplications option. It will list the available applications
        bootstrapCommandLineOptions.addOption("la", LIST_APPLICATIONS_OPTION, false,
                "List the available CCS applications in this distribution");
        // define the --application option
        bootstrapCommandLineOptions.addOption("app", APPLICATION_OPTION, true, "The APPLICATION to be launched");
        getOption(APPLICATION_OPTION).setArgName("APPLICATION");

        // The System property option
        Option sysProperty = OptionBuilder.withArgName("SystemProperty=Value").hasArgs(2)
                .withValueSeparator().withDescription("Set the Value of a SystemProperty.")
                .create("D");
        bootstrapCommandLineOptions.addOption(sysProperty);

        // Show the info on the current distribution
        bootstrapCommandLineOptions.addOption("di", "distInfo", false, "Show information on current distribution.");

        // Show property tree
        bootstrapCommandLineOptions.addOption("sp", "showProperties", true, "Show the properties for <FILE_NAME>.");
        getOption("showProperties").setArgName("FILE_NAME");

        // Show the classpath
        bootstrapCommandLineOptions.addOption("scp", "showClasspath", false, "Show the classpath for the given application.");
    }

    private static void resetBootstrap() {
        bootstrapCmdLineProperties = new Properties();
        additionalClassPathEntriesMap = new HashMap<>();
        additionalClassPathEntriesList = new ArrayList<>();
        bootstrapApplication = null;
        BootstrapUtils.reset();
    }

    public static void initializeBootstrap() {
        initializeBootstrap(BootstrapUtils.class);
    }

    /**
     * Initializes the Bootstrap Environment
     *
     * @param theClazz
     */
    public static void initializeBootstrap(Class theClazz) {
        resetBootstrap();
        Bootstrap.loaderClass = theClazz;
    }

    public static Class getLoaderClass() {
        return loaderClass;
    }

    /**
     * Get the properties for the current CCS application.
     *
     * @return The Properties for the current application.
     */
    public synchronized static Properties getBootstrapApplicationProperties() {
        if (bootstrapApplicationProperties == null) {
            bootstrapApplicationProperties = BootstrapUtils.getApplicationDefinitionProperties(getBootstrapApplication());
        }
        return bootstrapApplicationProperties;
    }

    protected List<String> getAdditionalCommandLineArguments() {
        return additionalCommandLineArguments;
    }
    
    /**
     * Parse the command line arguments and extract the needed information. Any
     * additional arguments will be passed as command line arguments to the CCS
     * application to be launched.
     *
     * @param args The command line arguments.
     * @param fromMain Indicates if the method is invoked from the main method,
     * i.e. if we are parsing the main cmd line. This method is invoked a second
     * time to extract additional system properties from the arguments passed to
     * the main class.
     * @throws ParseException if there are problems parsing the command line.
     *
     */
    protected CommandLine parseCommandLineArguments(String[] args) throws ParseException {

        //Check for arguments that start with -D and don't have value associated with it.
        //i.e. arguments of the form: -D some.prop
        //Also detect if there are any command line options that don't belong to 
        //the bootstrap. If any is found they are stored to be passed to the main application.
        List<String> argumentsToParse = new ArrayList<>();
        boolean skipNext = false;
        for (int i = 0; i < args.length; i++) {
            if ( skipNext ) {
                skipNext = false;
                continue;
            }
            String arg = args[i];
            if (arg.startsWith("-D")) {
                // Check if it's a "-D", i.e. there is a space "-D prop=value"
                int propertyStrIndex = arg.equals("-D") ? i + 1 : i;
                String propertyStr = args[propertyStrIndex].replace("-D", "");
                // Check if there is a property separator, either = or :
                if (!propertyStr.contains("=") /*&& ! propertyStr.contains(":")*/) {
                    args[propertyStrIndex] = args[propertyStrIndex].replace(propertyStr, propertyStr + "=");
                }
                if (arg.equals("-D")) {
                    argumentsToParse.add(arg);
                } else {
                    additionalCommandLineArguments.add(args[i]);
                }
            } else {
                if ( arg.startsWith("-") && ! bootstrapCommandLineOptions.hasOption(arg) ) {
                    //This is an option that does not belong to the bootstrap.
                    additionalCommandLineArguments.add(arg);
                    //Check if the option is followed by another argument
                    if ( i < args.length -1 ) {
                        String tmpArg = args[i+1];
                        if ( ! tmpArg.startsWith("-") ) {
                            additionalCommandLineArguments.add(tmpArg);
                            skipNext = true;
                        }
                    }
                } else {
                    argumentsToParse.add(arg);
                }
                
            }
        }
        

        String[] newArgs = argumentsToParse.toArray(new String[argumentsToParse.size()]);
        
        CommandLineParser parser = new BasicParser();
        CommandLine line = parser.parse(bootstrapCommandLineOptions, newArgs, false);
        
        if (line.hasOption(APPLICATION_OPTION)) {
            String tmpApp = line.getOptionValue(APPLICATION_OPTION);
            if ("true".equals(System.getProperty("org.lsst.ccs.bootstrap.test")) && !BootstrapUtils.getBootstrapListOfApplications().contains(tmpApp)) {
                throw new IllegalArgumentException("Application name:  " + tmpApp + " is not a valid value.");
            } else {
                bootstrapApplication = tmpApp;
            }
        }

        if (line.hasOption(HELP_OPTION)) {
            printHelp = true;
        }

        additionalCommandLineArguments.addAll(line.getArgList());

        showClasspath = line.hasOption("showClasspath");

        if (line.hasOption("showProperties")) {
            if (showProperties == null) {
                showProperties = line.getOptionValue("showProperties");
            }
        }

        if (line.hasOption(LIST_APPLICATIONS_OPTION)) {
            listApplications = true;
        }

        verbose = line.hasOption(VERBOSE_OPTION);

        if (printHelp && !passedAlongOptions.contains("-" + HELP_OPTION)) {
            passedAlongOptions.add("-" + HELP_OPTION);
        }
        if (verbose && !passedAlongOptions.contains("-" + VERBOSE_OPTION)) {
            passedAlongOptions.add("-" + VERBOSE_OPTION);
        }

        //Set the command line properties in the System Properties
        Properties cmdLineProperties = line.getOptionProperties("D");

        // Commons CLI sets properties values to "true" by default if such properties don't have a value.
        // This is overwritten in the following lines, by checking the user provided value
        // substituting it in the Properties object if needed.
        Option[] opts = line.getOptions();
        for (Option opt : opts) {
            if (opt.getOpt().equals("D")) {
                boolean wasSet = (opt.getValuesList().size() == 2);
                if (!wasSet) {
                    cmdLineProperties.setProperty(opt.getValue(), "");
                }
            }
        }

        if (verbose() && !quiet) {
            if (!cmdLineProperties.isEmpty()) {
                System.out.println("\n*** Adding the following command line properties to the Properties:");
                Set keys = cmdLineProperties.keySet();
                for (Object key : keys) {
                    System.out.println("\t" + key + " = " + cmdLineProperties.getProperty((String) key));
                }
            }
        }
        if (!cmdLineProperties.isEmpty()) {
            bootstrapCmdLineProperties.putAll(cmdLineProperties);
            System.getProperties().putAll(cmdLineProperties);
        }

        if (line.hasOption("distInfo")) {
            showDistributionInfo = true;
        }

        // Additional Command line arguments must be inspected to catch properties
        // passed with -D withoug having a space in it. See BootstrapParser for code
        // that lets such arguments pass the parsing phase.
        List<String> toBeRemoved = new ArrayList<>();
        for (String additional : additionalCommandLineArguments) {
            SystemPropertyMatcher m = SystemPropertyMatcher.matcher(additional);
            if (m.matches()) {
                toBeRemoved.add(additional);
                bootstrapCmdLineProperties.put(m.getProperty(), m.getValue());
            }
        }
        for (String remove : toBeRemoved) {
            additionalCommandLineArguments.remove(remove);
        }


        return line;
    }

    public static Properties getCmdLineProperties() {
        return bootstrapCmdLineProperties;
    }

    /**
     * The the name of the CCS application to be launched.
     *
     * @return The application name.
     */
    public static String getBootstrapApplication() {
//        if ( bootstrapApplication == null) {
//            throw new RuntimeException("The Bootstrap Application has not been defined yet.");
//        }
        return bootstrapApplication;
    }

    /**
     * Return true/false if help was requested.
     *
     * @return true/false if help was requested.
     */
    public boolean doPrintHelp() {
        return printHelp;
    }

    /**
     * Return true/false is verbose is turned on.
     *
     * @return true/false if verbose is turned on.
     */
    public static boolean verbose() {
        return verbose;
    }

    public static boolean isQuiet() {
        return quiet;
    }

    /**
     * Get the bootstrap command line options.
     *
     * @return The Options containing the command line options.
     *
     */
    protected Options getBootstrapCommandLineOptions() {
        return bootstrapCommandLineOptions;
    }

    /**
     * Prints the content of a URLClassLoader.
     *
     * @param classLoader
     */
    private static void printBootstrapClassLoader() {
        URL[] urls = getBootstrapApplicationClassLoader().getURLs();
        System.out.println("*** CLASSPATH");
        for (URL url : urls) {
            System.out.println("\t\t" + url);
        }
    }

    /**
     * Get a given command line Option
     *
     * @param opt The option name.
     * @return The corresponding Option.
     */
    private Option getOption(String opt) {
        return getBootstrapCommandLineOptions().getOption(opt);
    }

    /**
     * Get the list of arguments to be passed to the application. This list
     * contains options that are transfered to the application (like -help and
     * -verbose), the list of arguments defined in the application property file
     * definition and all the arguments that are left after parsing.
     *
     * @param appProperties
     * @return
     */
    private String[] getApplicationArguments(Properties appProperties) throws ParseException {

        String applicationArgs = appProperties.getProperty(APPLICATION_ARGS_PROPERTY, "").trim();
        StringTokenizer applicationArgsTokenizer = new StringTokenizer(applicationArgs, " ");

        int nArgs = passedAlongOptions.size() + additionalCommandLineArguments.size() + applicationArgsTokenizer.countTokens();

        String[] mainArgs = new String[nArgs];

        int argCount = 0;
        while (applicationArgsTokenizer.hasMoreTokens()) {
            mainArgs[argCount++] = applicationArgsTokenizer.nextToken();
        }
        for (String opt : passedAlongOptions) {
            mainArgs[argCount++] = opt;
        }
        for (String cmdArg : additionalCommandLineArguments) {
            mainArgs[argCount++] = cmdArg;
        }
        if (verbose() && !isQuiet()) {
            System.out.print("*** Command line arguments passed to the mainClass: ");
            for (String arg : mainArgs) {
                System.out.print(arg + " ");
            }
            System.out.println();
        }

        return mainArgs;
    }

    /**
     * Get the classLoader for this application.
     *
     * @return The URLClassLoader.
     */
    public static URLClassLoader getBootstrapApplicationClassLoader() {
        if (applicationClassLoader == null) {
            buildBootstrapClassLoader();
        }
        return applicationClassLoader;
    }

    /**
     * Scan a given Properties file to search for properties that end with
     * "additional.classpath.entry". Do we enforce uniqueness? Do we load them
     * all?
     *
     * @param props
     */
    static void scanPropertiesForClassPathEntries(Properties props) {

        Set<Object> keySet = BootstrapResourceUtils.getAllKeysInProperties(props);
        for (Object obj : keySet) {
            String key = (String) obj;
            if (key.endsWith("additional.classpath.entry")) {
                String entry = props.getProperty(key);
                if (additionalClassPathEntriesMap.containsKey(key)) {
                    if (!quiet) {
                        System.out.println("*** [WARNING] ignoring additional classpath entry: " + key + "=" + entry + " it was already added to the CLASSPATH as " + additionalClassPathEntriesMap.get(key));
                    }
                } else {
                    additionalClassPathEntriesMap.put(key, entry);
                    additionalClassPathEntriesList.add(entry);
                }
            }
        }
    }

    private synchronized static void buildBootstrapClassLoader() {
//        if (bootstrapApplication == null) {
//            throw new RuntimeException("The Bootstrap Application has not been defined yet.");
//        }

        if (applicationClassLoader != null) {
            throw new RuntimeException("The Bootstrap ClassLoader has already been built. Please report this problem.");
        }

        //Build the new classloader to launch the CCS application
        List<URL> classPathUrlList = new ArrayList<>();

        //Put all additional classpath entries from application definition file to classpath
        if (bootstrapApplication != null) {
            scanPropertiesForClassPathEntries(getBootstrapApplicationProperties());
        }

        try {

            String userProvidedDistributionDirs = BootstrapUtils.getUserProvidedDistributionDirectories();
            List<String> additionalDistributionJarDirs = BootstrapUtils.extractDirectoriesFromPath(userProvidedDistributionDirs, null, true);

            boolean checkForVersionIncompatibilities = additionalDistributionJarDirs.size() > 0;
            ClassPathBuilderSupport cpSupport = new ClassPathBuilderSupport();

            //First add the distribution main jar file
            String applicationMainJar = getDistributionMainJar();
            File jar = null;
            if (applicationMainJar != null) {
                jar = new File(applicationMainJar);
                classPathUrlList.add(jar.toURI().toURL());
            }

            //Before moving this code, check https://jira.slac.stanford.edu/browse/LSSTCCS-284
            //Add all the additional Entries to the classpath read from property files.        
            for (String classPathEntry : additionalClassPathEntriesList) {
                File cpEntryFile = new File(classPathEntry);
                if (cpEntryFile.isDirectory()) {
                    if (!quiet) {
                        System.out.println("*** [WARNING] Directories cannot be added to the classpath as additional Classpath entries. Skipping " + classPathEntry);
                    }
                } else {
                    classPathUrlList.add(cpEntryFile.toURI().toURL());
                }
            }

            if (checkForVersionIncompatibilities && jar != null) {
                scanManifestForClassPathElements(jar, cpSupport);
            }

            //Then add the distribution main jar for any additional distribution
            for (String dir : additionalDistributionJarDirs) {
                String additionalMainJar = getDistributionMainJar(dir);
                File mainjar = new File(additionalMainJar);
                classPathUrlList.add(mainjar.toURI().toURL());
                if (checkForVersionIncompatibilities) {
                    scanManifestForClassPathElements(mainjar, cpSupport);
                }
            }

        } catch (MalformedURLException mue) {
            throw new RuntimeException("Failed to build URL when building the classpath: " + mue.getMessage());
        }

        //Build the array of classpath URLs
        URL[] classpathUrls = new URL[classPathUrlList.size()];
        int count = 0;
        for (URL classpathURl : classPathUrlList) {
            classpathUrls[count++] = classpathURl;
        }

        applicationClassLoader = new URLClassLoader(classpathUrls);
    }

    /**
     * Get the fully qualified path to the distribution main jar from the
     * distribution definition file.
     *
     * @param distribution The distribution
     *
     * @return The main jar file path.
     */
    private static String getDistributionMainJar(String distribution) {
        File distributionDefinitionFile = new File(BootstrapUtils.getDistributionResourcesDirectory(distribution) + "DIST-INF/distribution.xml");
        if ( !distributionDefinitionFile.exists()) {
            if ( "true".equals(System.getProperty("org.lsst.ccs.bootstrap.test")) ) {
                return null;
            }
            throw new RuntimeException("FATAL: Cannot run distribution " + distribution + " as it does not contain the distribution definition xml file");
        }

        try {
            DocumentBuilderFactory dbFactory = DocumentBuilderFactory.newInstance();
            DocumentBuilder dBuilder = dbFactory.newDocumentBuilder();

            Document doc;
            doc = dBuilder.parse(distributionDefinitionFile);

            Element rootEl = doc.getDocumentElement();

            return BootstrapUtils.getDistributionJarFilesDirectory(distribution) + rootEl.getAttribute("mainJar");

        } catch (ParserConfigurationException | SAXException | IOException pce) {
            throw new RuntimeException(pce);
        }
    }

    private static String getDistributionMainJar() {
        if (distributionMainJar == null) {
            distributionMainJar = getDistributionMainJar(BootstrapUtils.getCCSDistributionRootDirectory());
        }
        return distributionMainJar;
    }

    /**
     * Launch the CCS Application for the given name.
     *
     * @param applicationName The name of the CCS application.
     */
    private void launchCCSApplication(String applicationName) {

        Properties applicationProperties = BootstrapUtils.getApplicationDefinitionProperties(applicationName);

        if (doPrintHelp()) {
            String appDescription = applicationProperties.getProperty(APPLICATION_DESCRIPTION_PROPERTY, "");
            System.out.println("\n\tCCS Application " + applicationName + " " + appDescription + "\n");
        }

        //Get the main Class to launch.
        String applicationMainClassName = applicationProperties.getProperty(BootstrapUtils.APPLICATION_MAINCLASS_PROPERTY);

        if (applicationMainClassName == null) {
            throw new RuntimeException("*** Application " + applicationName + " must contain define the main Class to lauch. "
                    + "This can be done either in the Manifest of the main jar or by defining the property "
                    + BootstrapUtils.APPLICATION_MAINCLASS_PROPERTY
                    + " in its definition file. ");
        }

        if (verbose() && isQuiet()) {
            System.out.println("*** Distribution Root: " + BootstrapUtils.getCCSDistributionRootDirectory());
            System.out.println("*** Application name: " + applicationName);
            System.out.println("*** MainClass: " + applicationMainClassName);
            System.out.println("*** LD_LIBRARY_PATH: " + System.getenv("LD_LIBRARY_PATH"));
        }

        // The LD_LIBRARY_PATH environment variable is now set in the CCSbootstrap.sh script.
        //Set up the LD_LIBRARY_PATH
//        try {
//            //The following is a hack to make it work. 
//            //What follows is from http://blog.cedarsoft.com/2010/11/setting-java-library-path-programmatically/
//            //the only way to set the property java.libary.path is to add a system property *before* the application is started:
//            //
//            //java -Djava.library.path=/path/to/libs
//            //
//            //Changing the system property later doesn’t have any effect, since the property is evaluated very early and cached.
//            //But the Classloader has a static field (sys_paths) that contains the paths. 
//            //If that field is set to null, it is initialized automatically. 
//            //Therefore forcing that field to null will result into the reevaluation of the library path as soon as loadLibrary() is called…
//            System.setProperty("java.library.path", BootstrapUtils.getDistributionJniDirectory() + File.pathSeparator + BootstrapUtils.getDistributionLibDirectory());
//            Field fieldSysPath = ClassLoader.class.getDeclaredField("sys_paths");
//            fieldSysPath.setAccessible(true);
//            fieldSysPath.set(null, null);
//        } catch (NoSuchFieldException | SecurityException | IllegalArgumentException | IllegalAccessException e) {
//            System.out.println("*** Failed to set java.library.path");
//        }
        try {
            Class applicationMainClass = Class.forName(applicationMainClassName);
            try {
                //Get the application main method.
                Method applicationMainMethod = applicationMainClass.getMethod("main",
                        new Class[]{String[].class});

                //Build the array containing the arguments to be passed to the main
                String[] mainArgs = getApplicationArguments(applicationProperties);

                //Print the classpath
                if (verbose() && !isQuiet()) {
                    printBootstrapClassLoader();
                }

                try {
                    //Invoke the main method on of the main Class.
                    applicationMainMethod.invoke(null, new Object[]{mainArgs});
                } catch (IllegalAccessException | IllegalArgumentException | InvocationTargetException e) {
                    System.out.println("*** Failed to invoke main method in class " + applicationMainClassName + "\n" + e.getMessage());
                    throw new RuntimeException(e);
                }

            } catch (NoSuchMethodException | SecurityException | ParseException e) {
                System.out.println("*** Could not access the main method in class " + applicationMainClassName + "\n" + e.getMessage());
            }

        } catch (ClassNotFoundException cnfe) {
            System.out.println("*** Could not find class " + applicationMainClassName + " in the following classpath: ");
            printBootstrapClassLoader();
            System.out.println("*************************************************************************\n" + cnfe.getMessage());
        }

    }

    public static void main(String[] args) throws Exception {

        System.setProperty(BOOTSTRAP_ENVIRONMENT_PROPERTY, "true");
        Bootstrap bootstrap = new Bootstrap();
        bootstrap.parseCommandLineArguments(args);

        String applicationName = getBootstrapApplication();

        if (bootstrap.showClasspath) {
            if (getBootstrapApplication() == null) {
                throw new IllegalArgumentException("To display the content of the classpath you have to provide an application with the --app option");
            } else {
                printBootstrapClassLoader();
            }
        } else if (bootstrap.showProperties != null) {
            if (getBootstrapApplication() == null) {
                throw new IllegalArgumentException("To display the content of a properties file you have to provide an application with the --app option");
            } else {
//                ResourcesUtils.printProperties(BootstrapUtils.getMergedProperties(bootstrap.showProperties));
                ResourcesUtils.printProperties(BootstrapResourceUtils.getBootstrapProperties(bootstrap.showProperties));
            }
        } else if (bootstrap.showDistributionInfo) {
            printDistributionInfo();
        } else if (bootstrap.listApplications) {

            List<String> availableApplications = BootstrapUtils.getBootstrapListOfApplications();
            if (availableApplications.isEmpty()) {
                System.out.println("No CCS applications are defined in the current distribution.");
                if (bootstrap.verbose() && !bootstrap.isQuiet()) {
                    System.out.println(BootstrapUtils.getDistributionResourcesDirectory());
                }
            } else {
                System.out.println("Available CCS applications :");
                if (bootstrap.verbose() && !bootstrap.isQuiet()) {
                    System.out.println(BootstrapUtils.getDistributionResourcesDirectory());
                }
                for (String application : availableApplications) {
                    Properties applicationProps = BootstrapUtils.getApplicationDefinitionProperties(application);
                    System.out.println("\t" + application + "\t" + applicationProps.getProperty(APPLICATION_DESCRIPTION_PROPERTY));
                }
            }

        } else if (bootstrap.doPrintHelp() && applicationName == null) {
            printHelp(bootstrap.getBootstrapCommandLineOptions());
        } else if (applicationName != null) {
            bootstrap.launchCCSApplication(applicationName);
        } else {
            printHelp(bootstrap.getBootstrapCommandLineOptions());
        }

    }

    private static void printHelp(Options o) {
        HelpFormatter formatter = new HelpFormatter();
        formatter.printHelp(100, "CCSbootstrap", "", o, "", true);
    }

    private static void printDistributionInfo() {
        System.out.println("\n*** Distribution info");
        System.out.println("\tDistribution path: " + BootstrapUtils.getCCSDistributionRootDirectory());
        System.out.println("\tResources ordered search path: ");
        List<String> distSearchPathList = BootstrapUtils.getOrderedListOfResourceDirectories();
        String resourcesDirList = "";
        StringBuffer b = new StringBuffer();
        for (String dir : distSearchPathList) {
            b.append("\t\t" + dir + "\n");
        }
        resourcesDirList += b.toString();
        System.out.print(resourcesDirList);

    }

    /*
     * This method is used to identify if we are running within the bootstrap environment.
     * It should be used to make sure that WARNING/ERROR messages that are related to the 
     * bootstrap environment don't propagate outside of it.
     * See https://jira.slac.stanford.edu/browse/LSSTCCS-209
     * 
     */
    public static boolean isBootstrapEnvironment() {
        return "true".equals(System.getProperty(BOOTSTRAP_ENVIRONMENT_PROPERTY));
    }

    private static void scanManifestForClassPathElements(File file, ClassPathBuilderSupport cp) {
        JarFile jarFile = null;
        try {
            jarFile = new JarFile(file);
            Manifest manifest = jarFile.getManifest();
            String manifestClassPath = manifest.getMainAttributes().getValue("Class-Path").trim();
            File parentDir = file.getParentFile();
            StringTokenizer classPathTokens = new StringTokenizer(manifestClassPath, " ");
            while (classPathTokens.hasMoreTokens()) {
                String manifestClassPathJar = classPathTokens.nextToken();
                cp.addClasspathEntry(parentDir.getAbsolutePath(), manifestClassPathJar);
            }
        } catch (IOException ioe) {
            throw new RuntimeException(ioe);
        } finally {
            if (jarFile != null) {
                try {
                    jarFile.close();
                } catch (IOException e) {
                    throw new RuntimeException(e);
                }
            }
        }

    }

    private static class ClassPathBuilderSupport {

        private HashMap<String, ClassPathElementWithVersion> classPathEntries = new HashMap<>();

        void addClasspathEntry(String resourceDir, String fileName) {

            String jarName = VersionComparator.stripVersion(fileName);

            ClassPathElementWithVersion existingEntry = classPathEntries.get(jarName);
            ClassPathElementWithVersion newEntry = new ClassPathElementWithVersion(resourceDir, fileName);
            if (existingEntry == null) {
                classPathEntries.put(jarName, newEntry);
            } else if (!existingEntry.fileName.equals(newEntry.fileName) && !quiet) {
                System.out.println("WARNING Classpath entry version conflict: ");
                System.out.println("\t " + newEntry.resourceDir + BootstrapUtils.FILE_SEPARATOR + newEntry.fileName);
                System.out.println("\t " + existingEntry.resourceDir + BootstrapUtils.FILE_SEPARATOR + existingEntry.fileName);
            }
        }

        Collection<ClassPathElementWithVersion> getListOfClasspathElements() {
            return classPathEntries.values();
        }
    }

    private static class ClassPathElementWithVersion {

        private String fileName, resourceDir;
        private URL elementURL;
        private Version version;

        public ClassPathElementWithVersion(String resourceDir, String fileName) {

            this.fileName = fileName;
            this.resourceDir = resourceDir;

        }

        Version getVersion() {
            if (version == null) {
                version = VersionComparator.getVersionFromFileName(fileName);
            }
            return version;
        }

        String getFileName() {
            return fileName;
        }

        URL getElementUrl() {
            File element = new File(resourceDir, fileName);
            if (element.exists()) {
                try {
                    return element.toURI().toURL();
                } catch (Exception e) {
                    e.printStackTrace();
                    return null;
                }
            } else {
                System.out.println("The following file does not exist : " + element.getAbsolutePath());
                System.out.println("Something went wrong with its version in the Bootstrap");
                return null;
            }
        }
    }
}
