package org.lsst.ccs.bootstrap.resources;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.ListIterator;
import java.util.Properties;
import java.util.Set;
import java.util.StringTokenizer;
import org.lsst.ccs.bootstrap.Bootstrap;
import org.lsst.ccs.bootstrap.BootstrapUtils;

/**
 * Utility class for handling resources.
 *
 * @author turri
 */
public class ResourcesUtils {

    private static final String FILE_SEPARATOR = System.getProperty("file.separator");

    /**
     * Get the list of resources in the given ResoucesTree for a given
     * extension.
     *
     * @param tree
     * @param extension The extension.
     * @param depth The depth of the recursive search. Negative number means all the way,
     * zero means right in the resouce directory.
     * @return
     */
    public static List<String> getResourcesInResourcesTreeByExtension(ResourcesTree tree, String extension, int depth) {
        List<String> resources = new ArrayList();
        for (ResourceDirectory dir : tree.getResourceDirectoryList()) {
            for (String resource : dir.getResources()) {
                if (extension != null && !resource.endsWith("." + extension)) {
                    continue;
                }

                if (depth > -1) {
                    StringTokenizer t = new StringTokenizer(resource, FILE_SEPARATOR);
                    if (t.countTokens() > depth + 1) {
                        continue;
                    }
                }

                if (!resources.contains(resource)) {
                    resources.add(resource);
                }
            }
        }
        return resources;
    }

    /**
     * Get a list of all the resources in a given ResourcesTree.
     *
     * @param tree The ResourcesTree
     * @return The List containing the name of all the resources.
     */
    public static List<String> getAllResourcesInResourcesTree(ResourcesTree tree) {
        return getResourcesInResourcesTreeByExtension(tree, null, -1);
    }

    /**
     * Get a merged version of a given property file from the given
     * ResourcesTree. The resource directories are scanned from the bottom up.
     * Every time a property file for the given name is found, its properties
     * are loaded using the load method on the Properties object. So all the
     * properties are merged.
     *
     * @param tree The ResourcesTree in which property files are searched.
     * @param fileName The name of the property file to search for.
     * @return The Properties object with the merged properties.
     */
    public static Properties getMergedPropertyFile(ResourcesTree tree, String fileName) {
        return getMergedPropertyFile(tree, fileName, false);
    }

    /**
     * Get a merged version of a given property file from the given
     * ResourcesTree. The resource directories are scanned from the bottom up.
     * Every time a property file for the given name is found, its properties
     * are loaded using the load method on the Properties object. So all the
     * properties are merged.
     *
     * @param tree The ResourcesTree in which property files are searched.
     * @param fileName The name of the property file to search for.
     * @param useSystem If true the System Properties will be the parent of the
     * returned Properties object.
     * @return The Properties object with the merged properties.
     */
    public static Properties getMergedPropertyFile(ResourcesTree tree, String fileName, boolean useSystem) {
        return getMergedProperties(tree, new String[]{fileName}, useSystem, null);

    }

    public static Properties getMergedProperties(ResourcesTree tree, String[] properties, boolean useSystem, ResourcesTreeProperties parentProps) {

        //First load the System properties is useSystem is set to true
        ResourcesTreeProperties props = parentProps;
        if (useSystem) {
            if (Bootstrap.verbose() && !Bootstrap.isQuiet()) {
                System.out.println("*** Adding System Properties to the properties chain");
            }
            props = new ResourcesTreeProperties("SystemProperties", null, props);
            props.putAll(System.getProperties());
        }

        //Then load the properties from ccsDefaults.properties in the distribution resource directory
        try {
            InputStream defaultsIn = BootstrapUtils.class.getResourceAsStream("/ccsDefaults.properties");
            if (defaultsIn != null) {
                if (Bootstrap.verbose() && !Bootstrap.isQuiet()) {
                    System.out.println("*** Adding Properties " + BootstrapUtils.getDistributionResourcesDirectory() + "ccsDefaults.properties to properties chain");
                }
                Properties parent = props;
                props = new ResourcesTreeProperties("ccsDefaults", BootstrapUtils.getDistributionResourcesDirectory(), parent);
                props.load(defaultsIn);
                String applicationName = Bootstrap.getBootstrapApplication();
                props.put("org.lsst.ccs.application.name", applicationName == null ? "" : applicationName);
            }
        } catch (IOException ioe) {
            throw new RuntimeException(ioe);
        }

        List<ResourceDirectory> resourceList = tree.getResourceDirectoryList();
        ListIterator<ResourceDirectory> reverseIterator = resourceList.listIterator(resourceList.size());
        while (reverseIterator.hasPrevious()) {
            ResourceDirectory dir = reverseIterator.previous();
            for (int i = 0; i < properties.length; i++) {

                String propertyFile = properties[i];
                if (propertyFile != null) {
                    if (!propertyFile.endsWith(".properties")) {
                        propertyFile += ".properties";
                    }

                    // Check if the property file exists either in its fully qualified path, or at the root
                    String fileLocation = null;
                    if (dir.hasResource(propertyFile)) {
                        fileLocation = propertyFile;
                    }
                    if (propertyFile.contains(FILE_SEPARATOR) && fileLocation == null) {
                        propertyFile = propertyFile.substring(propertyFile.lastIndexOf(FILE_SEPARATOR) + 1);
                        if (dir.hasResource(propertyFile)) {
                            fileLocation = propertyFile;
                        }
                    }

                    if (fileLocation != null) {
                        FileInputStream in = null;
                        try {
                            in = new FileInputStream(new File(dir.getResouceDirectoryPath(), fileLocation));
                            Properties parent = props;
                            props = new ResourcesTreeProperties(fileLocation, dir.getResouceDirectoryPath(), parent);
                            props.load(in);
                            if (Bootstrap.verbose() && !Bootstrap.isQuiet()) {
                                System.out.println("*** Adding Properties " + dir.getResouceDirectoryPath() + fileLocation + " to properties chain");
                            }
                        } catch (IOException ioe) {
                            throw new RuntimeException(ioe);
                        } finally {
                            if (in != null) {
                                try {
                                    in.close();
                                } catch (IOException e) {
                                    throw new RuntimeException(e);
                                }
                            }
                        }
                    }
                }
            }
        }
        return props;
    }

    public static void printProperties(Properties props) {
        System.out.println("*****************");
        if (props instanceof ResourcesTreeProperties) {
            ResourcesTreeProperties p = (ResourcesTreeProperties) props;
            String output = "Properties from " + p.getPropertyFileName();
            if (p.getResourceDirectory() != null) {
                output += " in resource directory " + p.getResourceDirectory();
            }
            System.out.println(output);
        } else {
            System.out.println("External System properties");
        }
        for (Object key : props.keySet()) {
            System.out.println("\t" + key + " = " + props.getProperty((String) key));
        }
        if (props instanceof ResourcesTreeProperties) {
            ResourcesTreeProperties p = (ResourcesTreeProperties) props;
            if (p.hasParent()) {
                printProperties(p.getParent());
            }
        }
    }

    public static void loadKeySetForProperties(Properties props, Set<Object> set) {
        Set<Object> propsKey = props.keySet();
        for (Object obj : propsKey) {
            if (!set.contains(obj)) {
                set.add(obj);
            }
        }
        if (props instanceof ResourcesTreeProperties && ((ResourcesTreeProperties) props).getParent() != null) {
            loadKeySetForProperties(((ResourcesTreeProperties) props).getParent(), set);
        }
    }

    public static Properties getFlatPropertiesObject(Properties props) {
        Properties newProps = new Properties();
        Set<Object> keys = BootstrapResourceUtils.getAllKeysInProperties(props);
        for (Object key : keys) {
            newProps.put(key, props.getProperty((String) key));
        }
        return newProps;
    }

    public static InputStream getResourceFromResourceTree(ResourcesTree tree, String resourceName) throws FileNotFoundException {
        List<ResourceDirectory> resourceList = tree.getResourceDirectoryList();

        for (ResourceDirectory dir : resourceList) {
            if (dir.hasResource(resourceName)) {
                if (Bootstrap.verbose() && !Bootstrap.isQuiet()) {
                    System.out.println("*** Found resource " + dir.getResouceDirectoryPath() + resourceName);
                }
                return new FileInputStream(dir.getResouceDirectoryPath() + resourceName);
            } else {
                if (resourceName.contains(FILE_SEPARATOR)) {
                    resourceName = resourceName.substring(resourceName.lastIndexOf(FILE_SEPARATOR) + 1);
                    if (dir.hasResource(resourceName)) {
                        if (Bootstrap.verbose() && !Bootstrap.isQuiet()) {
                            System.out.println("*** Found resource " + dir.getResouceDirectoryPath() + resourceName);
                        }
                        return new FileInputStream(dir.getResouceDirectoryPath() + resourceName);
                    }
                }
            }
        }

        return null;
    }

}
