package org.apache.commons.cli;

import java.util.Arrays;
import java.util.List;
import java.util.ListIterator;
import java.util.Properties;
import org.apache.commons.cli.Options;

/**
 *
 * @author turri
 */
public class BootstrapParser extends BasicParser {

    @Override
    protected void processOption(String arg, ListIterator iter) throws ParseException {
        boolean hasOption = getOptions().hasOption(arg);

        // if there is no option throw an UnrecognisedOptionException
        if (!hasOption) {
            cmd.addArg(arg);
            return;
        }

        // get the option represented by arg
        Option opt = (Option) getOptions().getOption(arg).clone();

        // update the required options and groups
        updateRequiredOptions(opt);

        // if the option takes an argument value
        if (opt.hasArg()) {
            processArgs(opt, iter);
        }

        // set the option on the command line
        cmd.addOption(opt);
    }

    private void updateRequiredOptions(Option opt) throws ParseException {
        // if the option is a required option remove the option from
        // the requiredOptions list
        if (opt.isRequired()) {
            getRequiredOptions().remove(opt.getKey());
        }

        // if the option is in an OptionGroup make that option the selected
        // option of the group
        if (getOptions().getOptionGroup(opt) != null) {
            OptionGroup group = getOptions().getOptionGroup(opt);

            if (group.isRequired()) {
                getRequiredOptions().remove(group);
            }

            group.setSelected(opt);
        }
    }

    @Override
    public void processArgs(Option opt, ListIterator iter) throws ParseException {
        // loop until an option is found
        while (iter.hasNext()) {
            String str = (String)(iter.next());

            // found an Option, not an argument
            if (getOptions().hasOption(str) && str.startsWith("-")) {
                iter.previous();
                break;
            }

            // found a value
            try {
                opt.addValueForProcessing(Util.stripLeadingAndTrailingQuotes(str));
            } catch (RuntimeException exp) {
                iter.previous();
                break;
            }
        }

        if (opt.getValues() == null && !opt.hasOptionalArg()) {
            throw new MissingArgumentException(opt);
        }
    }

    public CommandLine parse(Options options, String[] arguments, Properties properties, boolean stopAtNonOption)
            throws ParseException {
        // clear out the data in options in case it's been used before (CLI-71)
        for (Object opt : options.helpOptions()) {
            ((Option) opt).clearValues();
        }

        // clear the data from the groups
        for (Object group : options.getOptionGroups()) {
            ((OptionGroup) group).setSelected(null);
        }

        // initialise members
        setOptions(options);

        cmd = new CommandLine();

        boolean eatTheRest = false;

        if (arguments == null) {
            arguments = new String[0];
        }

        List<String> tokenList = Arrays.asList(flatten(getOptions(), arguments, stopAtNonOption));

        ListIterator<String> iterator = tokenList.listIterator();

        // process each flattened token
        while (iterator.hasNext()) {
            String t = iterator.next();
            // the value is the double-dash
            if ("--".equals(t)) {
                eatTheRest = true;
            } // the value is a single dash
            else if ("-".equals(t)) {
                if (stopAtNonOption) {
                    eatTheRest = true;
                } else {
                    cmd.addArg(t);
                }
            } // the value is an option
            else if (t.startsWith("-")) {
                if (stopAtNonOption && !getOptions().hasOption(t)) {
                    eatTheRest = true;
                    cmd.addArg(t);
                } else {
                    processOption(t, iterator);
                }
            } // the value is an argument
            else {
                cmd.addArg(t);

                if (stopAtNonOption) {
                    eatTheRest = true;
                }
            }

        }

        processProperties(properties);
        checkRequiredOptions();

        return cmd;
    }
}
