package org.lsst.ccs.bootstrap;

import java.io.File;
import java.net.URL;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;
import java.util.StringTokenizer;
import org.lsst.ccs.bootstrap.resources.BootstrapResourceUtils;
import org.lsst.ccs.bootstrap.resources.ResourcesTree;
import org.lsst.ccs.bootstrap.resources.ResourcesUtils;

/**
 * Utility methods for the Bootstrap application.
 *
 * @author turri
 */
public abstract class BootstrapUtils {

    public static final String FILE_SEPARATOR = System.getProperty("file.separator");
    private static final String PATH_SEPARATOR = System.getProperty("path.separator");
    private static final String DISTRIBUTION_JARS_DIRECTORY = "share" + FILE_SEPARATOR + "java" + FILE_SEPARATOR;
    private static final String DISTRIBUTION_RESOURCES_DIRECTORY = "etc" + FILE_SEPARATOR;
    private static final String DISTRIBUTION_LIB_DIRECTORY = "lib" + FILE_SEPARATOR;
    private static final String DISTRIBUTION_JNI_DIRECTORY = DISTRIBUTION_LIB_DIRECTORY + "jni" + FILE_SEPARATOR;
    static final String APPLICATION_MAINJAR_PROPERTY = "org.lsst.ccs.application.mainJar";
    private static final String TRANSPORT_DEFINITION_PROPERTY = "lsst.ccs.transport";
    private static List<String> listOfApplications = null, listOfTransports = null;
    /**
     * Environment variable that defines the resource path.
     */
    final static String CCS_RESOURCE_PATH_ENV_VAR = "CCS_RESOURCE_PATH";
    private static String ccsDistributionRootDirectory = null;
    private static ResourcesTree resourcesTree = null;

    /**
     * Get the ResourceTree for this Bootstrap environment. It contains all the
     * resources defined in the resource directory list.
     *
     * @return
     */
    public static ResourcesTree getBootstrapResourcesTree() {
        if (resourcesTree == null) {
            resourcesTree = new ResourcesTree();
            List<String> resourceDirectories = getOrderedListOfResourceDirectories();
            for (String resourceDirectory : resourceDirectories) {
                resourcesTree.addResourceDirectory(resourceDirectory);
            }
        }
        return resourcesTree;
    }

    /**
     * Get the ordered list of resource directories.
     *
     * The built in default is: - Distribution resources directory:
     * distribution/etc/ - System resources directory: /etc/ccs/ - User
     * resources directory: ~/.ccs/etc/
     *
     * By setting the environment variable CCS_RESOURCE_PATH, the developer can
     * change which resource directories are searched and in which order.
     *
     * When the CCS_RESOURCE_PATH environment variable is set, the distribution
     * resource directory will be still searched first, before any of the
     * directories provided by the user.
     *
     * This list is built once and its result is cached.
     *
     * @return The list of ordered directories used to search for resources.
     *
     */
    static List<String> getOrderedListOfResourceDirectories() {
        String userProvidedResourceDirs = System.getenv(CCS_RESOURCE_PATH_ENV_VAR);
        return getOrderedListOfResourceDirectories(userProvidedResourceDirs, true);
    }

    /**
     * This method should do all the work of setting up the resource directories
     * as it is used in by the tests.
     *
     * @param userProvidedResourceDirs This resembles the environment variable
     * to set the list of resource directories.
     * @return The list of ordered directories to search for resources.
     *
     */
    public static List<String> getOrderedListOfResourceDirectories(String userProvidedResourceDirs, boolean checkExistance) {

        String resourceDirectories = getDistributionResourcesDirectory();
        if (userProvidedResourceDirs != null && !userProvidedResourceDirs.equals("")) {
            resourceDirectories = userProvidedResourceDirs + PATH_SEPARATOR + resourceDirectories;
        }

        List<String> orderedListOfResourceDirectories = new ArrayList<>();
        StringTokenizer dirsToken = new StringTokenizer(resourceDirectories, ":;");
        while (dirsToken.hasMoreTokens()) {
            String resourceDir = dirsToken.nextToken().trim();
            if (!resourceDir.endsWith(FILE_SEPARATOR)) {
                resourceDir += FILE_SEPARATOR;
            }


            //The root for relative paths is the root of the CCS distribution.
            Path p = Paths.get(BootstrapUtils.getCCSDistributionRootDirectory());
            resourceDir = resourceDir.replace("~", System.getProperty("user.home"));
            resourceDir = p.resolve(resourceDir).normalize().toString();
            if (!resourceDir.endsWith(FILE_SEPARATOR)) {
                resourceDir += FILE_SEPARATOR;
            }
            File dir = new File(resourceDir);
            if (checkExistance && !dir.exists()) {
//                throw new IllegalArgumentException("Resource directory " + resourceDir + " does not exist. "
//                        + "Please update the environment variable " + CCS_RESOURCE_PATH_ENV_VAR);
            } else {
                orderedListOfResourceDirectories.add(resourceDir);
            }
        }
        return orderedListOfResourceDirectories;
    }

    /**
     * Get the Distribution resources directory.
     *
     * @return The resources directory in the distribution.
     */
    static String getDistributionResourcesDirectory() {
        return getDistributionResourcesDirectory(getCCSDistributionRootDirectory());
    }

    static String getDistributionResourcesDirectory(String distribution) {
        if (!distribution.endsWith(FILE_SEPARATOR)) {
            distribution += FILE_SEPARATOR;
        }
        return distribution + DISTRIBUTION_RESOURCES_DIRECTORY;
    }

    /**
     * Get the lib directory in the distribution.
     *
     * @return the lib files directory.
     *
     */
    static String getDistributionLibDirectory() {
        return getDistributionLibDirectory(getCCSDistributionRootDirectory());
    }

    static String getDistributionLibDirectory(String distribution) {
        if (!distribution.endsWith(FILE_SEPARATOR)) {
            distribution += FILE_SEPARATOR;
        }
        return distribution + DISTRIBUTION_LIB_DIRECTORY;
    }

    /**
     * Get the jni directory in the distribution.
     *
     * @return the jni files directory.
     *
     */
    static String getDistributionJniDirectory() {
        return getDistributionJniDirectory(getCCSDistributionRootDirectory());
    }

    static String getDistributionJniDirectory(String distribution) {
        if (!distribution.endsWith(FILE_SEPARATOR)) {
            distribution += FILE_SEPARATOR;
        }
        return distribution + DISTRIBUTION_JNI_DIRECTORY;
    }

    /**
     * Get the jar files directory in the distribution.
     *
     * @return the jar files directory.
     *
     */
    static String getDistributionJarFilesDirectory() {
        return getDistributionJarFilesDirectory(getCCSDistributionRootDirectory());
    }

    static String getDistributionJarFilesDirectory(String distribution) {
        if (!distribution.endsWith(FILE_SEPARATOR)) {
            distribution += FILE_SEPARATOR;
        }
        return distribution + DISTRIBUTION_JARS_DIRECTORY;
    }

    /**
     * Get the root directory of the CCS distribution from which the Bootstrap
     * code has been launched.
     *
     * @return The root directory of the CCS distribution. It ends with a
     * file.separator.
     */
    public static String getCCSDistributionRootDirectory() {
        if (ccsDistributionRootDirectory == null) {
            ccsDistributionRootDirectory = getCCSDistributionRootDirectory(BootstrapUtils.class);
        }
        return ccsDistributionRootDirectory;
    }

    /**
     * This is to be used only in test environment.
     *
     * @param clazz
     * @return
     */
    static String getCCSDistributionRootDirectory(Class clazz) {
        URL location = clazz.getProtectionDomain().getCodeSource().getLocation();
        String sourceCodeLocation = location.getFile();
        if (sourceCodeLocation.endsWith(".jar")) {
            ccsDistributionRootDirectory = sourceCodeLocation.substring(0, sourceCodeLocation.lastIndexOf(FILE_SEPARATOR) + 1);
        } else if (sourceCodeLocation.endsWith(FILE_SEPARATOR)) {
            ccsDistributionRootDirectory = sourceCodeLocation;
        } else {
            throw new RuntimeException("Could not process souce code location " + sourceCodeLocation + " It is neither a directory nor a jar file.");
        }

        if (ccsDistributionRootDirectory.endsWith(DISTRIBUTION_JARS_DIRECTORY)) {
            ccsDistributionRootDirectory = ccsDistributionRootDirectory.replace(DISTRIBUTION_JARS_DIRECTORY, "");
//            throw new RuntimeException("This executable is not being run from a CCS distribution directory. "+ccsDistributionRootDirectory);
        }
        return ccsDistributionRootDirectory;
    }

    /**
     * Get the list of available applications for the current Bootstrap
     * configuration.
     *
     * @return The list of application names.
     *
     */
    public static List<String> getBootstrapListOfApplications() {
        if (listOfApplications == null) {
            listOfApplications = new ArrayList<>();
            List<String> allPropertiesInResources = ResourcesUtils.getResourcesInResourcesTreeByExtension(getBootstrapResourcesTree(), "properties");
            for (String propertyFile : allPropertiesInResources) {
                Properties props = ResourcesUtils.getMergedPropertyFile(getBootstrapResourcesTree(), propertyFile);
                String mainJar = props.getProperty(APPLICATION_MAINJAR_PROPERTY);
                if (mainJar != null && !mainJar.isEmpty()) {
                    listOfApplications.add(propertyFile.replace(".properties", ""));
                }
            }
        }
        return listOfApplications;
    }

    /**
     * Get the list of available transport protocols for the current Bootstrap
     * configuration.
     *
     * @return The list of available transports.
     */
    static List<String> getBootstrapListOfTransports() {
        if (listOfTransports == null) {
            listOfTransports = new ArrayList<>();
            List<String> allPropertiesInResources = ResourcesUtils.getResourcesInResourcesTreeByExtension(getBootstrapResourcesTree(), "properties");
            for (String propertyFile : allPropertiesInResources) {
                Properties props = ResourcesUtils.getMergedPropertyFile(getBootstrapResourcesTree(), propertyFile);
                String transportProperty = props.getProperty(TRANSPORT_DEFINITION_PROPERTY);
                if (transportProperty != null) {
                    listOfTransports.add(propertyFile.replace(".properties", ""));
                }
            }
        }
        return listOfTransports;
    }

    /**
     * Get the transport definition properties file for a given transport name.
     *
     * @param transport The name of the application.
     * @return The Properties that define the application.
     */
    static Properties getPropertiesForTransport(String transport) {
//        return getMergedProperties(transport);
        return BootstrapResourceUtils.getBootstrapProperties(transport, false);
    }

    /**
     * Get the application definition properties file for a given application
     * name.
     *
     * @param application The name of the application.
     * @return The Properties that define the application.
     */
    static Properties getApplicationDefinitionProperties(String application) {
//        return getMergedProperties(application);
        return BootstrapResourceUtils.getBootstrapProperties(application, false);
    }

    static String getBootstrapLibraryPath() {
        String existingLibraryPath = System.getenv("LD_LIBRARY_PATH");
        String libraryPath = "";
        if (existingLibraryPath != null && !existingLibraryPath.isEmpty()) {
            libraryPath = existingLibraryPath + File.pathSeparator;
        }
        libraryPath += BootstrapUtils.getDistributionJniDirectory() + File.pathSeparator + BootstrapUtils.getDistributionLibDirectory();
        return libraryPath;
    }
}
