package org.lsst.ccs.bootstrap.resources;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.List;
import java.util.ListIterator;
import java.util.Properties;
import java.util.Set;

/**
 * Utility class for handling resources.
 *
 * @author turri
 */
public class ResourcesUtils {

    /**
     * Get the list of resources in the given ResoucesTree for a given
     * extension.
     *
     * @param tree
     * @param extension The extension.
     * @return
     */
    public static List<String> getResourcesInResourcesTreeByExtension(ResourcesTree tree, String extension) {
        List<String> resources = new ArrayList();
        for (ResourceDirectory dir : tree.getResourceDirectoryList()) {
            for (String resource : dir.getResources()) {
                if (extension != null && !resource.endsWith("." + extension)) {
                    continue;
                }
                if (!resources.contains(resource)) {
                    resources.add(resource);
                }
            }
        }
        return resources;
    }

    /**
     * Get a list of all the resources in a given ResourcesTree.
     *
     * @param tree The ResourcesTree
     * @return The List containing the name of all the resources.
     */
    public static List<String> getAllResourcesInResourcesTree(ResourcesTree tree) {
        return getResourcesInResourcesTreeByExtension(tree, null);
    }

    /**
     * Get a merged version of a given property file from the given
     * ResourcesTree. The resource directories are scanned from the bottom up.
     * Every time a property file for the given name is found, its properties
     * are loaded using the load method on the Properties object. So all the
     * properties are merged.
     *
     * @param tree The ResourcesTree in which property files are searched.
     * @param fileName The name of the property file to search for.
     * @return The Properties object with the merged properties.
     */
    public static Properties getMergedPropertyFile(ResourcesTree tree, String fileName) {
        return getMergedPropertyFile(tree, fileName, false);
    }

    /**
     * Get a merged version of a given property file from the given
     * ResourcesTree. The resource directories are scanned from the bottom up.
     * Every time a property file for the given name is found, its properties
     * are loaded using the load method on the Properties object. So all the
     * properties are merged.
     *
     * @param tree The ResourcesTree in which property files are searched.
     * @param fileName The name of the property file to search for.
     * @param useSystem If true the System Properties will be the parent of the
     * returned Properties object.
     * @return The Properties object with the merged properties.
     */
    public static Properties getMergedPropertyFile(ResourcesTree tree, String fileName, boolean useSystem) {
        return getMergedProperties(tree, new String[]{fileName}, useSystem);

    }

    public static Properties getMergedProperties(ResourcesTree tree, String[] properties, boolean useSystem) {
        Properties props = null;
        List<ResourceDirectory> resourceList = tree.getResourceDirectoryList();
        ListIterator<ResourceDirectory> reverseIterator = resourceList.listIterator(resourceList.size());
        while ( reverseIterator.hasPrevious() ) {
            ResourceDirectory dir = reverseIterator.previous();
            for (String propertyFile : properties) {
                if (propertyFile != null) {
                    if (!propertyFile.endsWith(".properties")) {
                        propertyFile += ".properties";
                    }
                    if (dir.hasResource(propertyFile)) {
                        try {
                            FileInputStream in = new FileInputStream(new File(dir.getResouceDirectoryPath(), propertyFile));
                            Properties parent = props != null ? props : useSystem ? System.getProperties() : null;
                            props = new ResourcesTreeProperties(propertyFile, dir.getResouceDirectoryPath(), parent);
                            props.load(in);
                            in.close();
                        } catch (IOException ioe) {
                            throw new RuntimeException(ioe);
                        }
                    }
                }
            }
        }
        return props;
    }

    public static void printProperties(Properties props) {
        System.out.println("*****************");
        if (props instanceof ResourcesTreeProperties) {
            ResourcesTreeProperties p = (ResourcesTreeProperties) props;
            String output = "Properties from " + p.getPropertyFileName();
            if (p.getResourceDirectory() != null) {
                output += " in resource directory " + p.getResourceDirectory();
            }
            System.out.println(output);
        } else {
            System.out.println("External System properties");
        }
        for (Object key : props.keySet()) {
            System.out.println("\t" + key + " = " + props.getProperty((String) key));
        }
        if (props instanceof ResourcesTreeProperties) {
            ResourcesTreeProperties p = (ResourcesTreeProperties) props;
            if (p.hasParent()) {
                printProperties(p.getParent());
            }
        }
    }

    public static Set<Object> getAllKeysInProperties(Properties props) {
        Set<Object> keySet = new HashSet<>();
        if (props != null) {
            loadKeySetForProperties(props, keySet);
        }
        return keySet;
    }

    private static void loadKeySetForProperties(Properties props, Set<Object> set) {
        Set<Object> propsKey = props.keySet();
        for (Object obj : propsKey) {
            if (!set.contains(obj)) {
                set.add(obj);
            }
        }
        if (props instanceof ResourcesTreeProperties && ((ResourcesTreeProperties) props).getParent() != null) {
            loadKeySetForProperties(((ResourcesTreeProperties) props).getParent(), set);
        }
    }
    
    public static Properties getFlatPropertiesObject(Properties props) {
        Properties newProps = new Properties();
        Set<Object> keys = getAllKeysInProperties(props);
        for (Object key : keys ) {
            newProps.put(key, props.getProperty((String)key));
        }
        return newProps;
    }
    
}
