package org.lsst.ccs.bootstrap;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.jar.JarFile;
import java.util.jar.Manifest;
import org.apache.commons.cli.BasicParser;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.OptionBuilder;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.lsst.ccs.bootstrap.resources.ResourcesUtils;

/**
 *
 * Bootstrap main class. Used to launch any CCS application.
 *
 * @author turri
 */
public class Bootstrap {

    // The Options containing the bootstrap command line options.
    private Options bootstrapCommandLineOptions;
    private List<String> additionalCommandLineArguments = new ArrayList<>(), passedAlongOptions = new ArrayList<>();
    // The name of the application to be launched.    
    private static String bootstrapApplication = null;
    public static URLClassLoader applicationClassLoader = null;
    private boolean printHelp = false, listApplications = false, verbose = false, showDistributionInfo = false;
    public static final String APPLICATION_OPTION = "application";
    public static final String HELP_OPTION = "help";
    public static final String VERBOSE_OPTION = "verbose";
    public static final String LIST_APPLICATIONS_OPTION = "listApplications";
    private String defaultTransport = null, transport, showProperties = null;
    private boolean hasTransport = false, showClasspath = false;
    private static final String APPLICATION_MAINCLASS_PROPERTY = "org.lsst.ccs.application.mainClass";
    private static final String APPLICATION_ARGS_PROPERTY = "org.lsst.ccs.application.args";
    private static final String APPLICATION_DESCRIPTION_PROPERTY = "org.lsst.ccs.application.description";
    private static Properties bootstrapCmdLineProperties = new Properties();
    private static boolean parsedOptions = false;
    private static Map<String, String> additionalClassPathEntriesMap = new HashMap<>();
    private static List<String> additionalClassPathEntriesList = new ArrayList<>();
    private static Properties bootstrapApplicationProperties = null;

    public Bootstrap() {
        // define the command line options
        bootstrapCommandLineOptions = new Options();
        // The help option
        bootstrapCommandLineOptions.addOption("h", HELP_OPTION, false, "Print the help message");
        // The verbose option
        bootstrapCommandLineOptions.addOption("v", VERBOSE_OPTION, false, "Turns on verbose statements");
        // The listApplications option. It will list the available applications
        bootstrapCommandLineOptions.addOption("la", LIST_APPLICATIONS_OPTION, false,
                "List the available CCS applications in this distribution");
        // define the --application option
        bootstrapCommandLineOptions.addOption("app", APPLICATION_OPTION, true, "The APPLICATION to be launched");
        getOption(APPLICATION_OPTION).setArgName("APPLICATION");

        // The System property option
        Option sysProperty = OptionBuilder.withArgName("SystemProperty=Value").hasArgs(2)
                .withValueSeparator().withDescription("Set the Value of a SystemProperty.")
                .create("D");
        bootstrapCommandLineOptions.addOption(sysProperty);

        // Show the info on the current distribution
        bootstrapCommandLineOptions.addOption("di", "distInfo", false, "Show information on current distribution.");

        // Show property tree
        bootstrapCommandLineOptions.addOption("sp", "showProperties", true, "Show the properties for <FILE_NAME>.");
        getOption("showProperties").setArgName("FILE_NAME");

        // Show the classpath
        bootstrapCommandLineOptions.addOption("scp", "showClasspath", false, "Show the classpath for the given application.");

        List<String> transports = BootstrapUtils.getBootstrapListOfTransports();
        if (!transports.isEmpty()) {
            hasTransport = true;
            bootstrapCommandLineOptions.addOption("tl", "transportLayer", true, "The TRANSPORT_LAYER to be used");
            getOption("transportLayer").setArgName("TRANSPORT_LAYER");

            String description = "The TRANSPORT_LAYER to be used. Possible values are: \n";
            for (String transport : transports) {
                description += transport + "\t";
                Properties transportProps = BootstrapUtils.getPropertiesForTransport(transport);
                if (transportProps.getProperty("org.lsst.ccs.transport.default") != null) {
                    defaultTransport = transport;
                }
            }
            description += "\nIf not specified the following transport will be used as the default: " + defaultTransport;
            getOption("transportLayer").setDescription(description);
        }

    }

    /**
     * Get the properties for the current CCS application.
     *
     * @return The Properties for the current application.
     */
    public static Properties getBootstrapApplicationProperties() {
        if ( bootstrapApplicationProperties == null ) {
            bootstrapApplicationProperties = BootstrapUtils.getApplicationDefinitionProperties(getBootstrapApplication());
        }
        return bootstrapApplicationProperties;
    }

    /**
     * Parse the command line arguments and extract the needed information. Any
     * additional arguments will be passed as command line arguments to the CCS
     * application to be launched.
     *
     * @param args The command line arguments.
     * @param fromMain Indicates if the method is invoked from the main method,
     * i.e. if we are parsing the main cmd line. This method is invoked a second
     * time to extract additional system properties from the arguments passed to
     * the main class.
     * @throws ParseException if there are problems parsing the command line.
     *
     */
    private void parseCommandLineArguments(String[] args) throws ParseException {

        if (parsedOptions) {
            throw new RuntimeException("Command line arguments can be parsed only once. Something went wrong ");
        } else {
            parsedOptions = true;
        }

        CommandLineParser parser = new BasicParser();
        CommandLine line = parser.parse(bootstrapCommandLineOptions, args, true);

        if (line.hasOption(APPLICATION_OPTION)) {
            if (bootstrapApplication != null) {
                throw new IllegalArgumentException("\t[FATAL] The application has been defined twice.");
            }
            String tmpApp = line.getOptionValue(APPLICATION_OPTION);
            if (!BootstrapUtils.getBootstrapListOfApplications().contains(tmpApp)) {
                throw new IllegalArgumentException("Application name:  " + tmpApp + " is not a valid value.");
            } else {
                bootstrapApplication = tmpApp;
            }
        }

        if (line.hasOption(HELP_OPTION)) {
            printHelp = true;
        }

        additionalCommandLineArguments.addAll(line.getArgList());

        showClasspath = line.hasOption("showClasspath");

        if (line.hasOption("showProperties")) {
            if (showProperties == null) {
                showProperties = line.getOptionValue("showProperties");
            }
        }

        if (line.hasOption(LIST_APPLICATIONS_OPTION)) {
            listApplications = true;
        }

        if (line.hasOption(VERBOSE_OPTION)) {
            verbose = true;
        }

        if (printHelp && !passedAlongOptions.contains("-" + HELP_OPTION)) {
            passedAlongOptions.add("-" + HELP_OPTION);
        }
        if (verbose && !passedAlongOptions.contains("-" + VERBOSE_OPTION)) {
            passedAlongOptions.add("-" + VERBOSE_OPTION);
        }

        //Set the command line properties in the System Properties
        Properties cmdLineProperties = line.getOptionProperties("D");
        if (verbose()) {
            if (!cmdLineProperties.isEmpty()) {
                System.out.println("\n*** Adding the following command line properties to the System Properties:");
                Set keys = cmdLineProperties.keySet();
                for (Object key : keys) {
                    System.out.println("\t" + key + " = " + cmdLineProperties.getProperty((String) key));
                }
            }
        }
        if (!cmdLineProperties.isEmpty()) {
            bootstrapCmdLineProperties.putAll(cmdLineProperties);
        }

        if (line.hasOption("distInfo")) {
            showDistributionInfo = true;
        }

        if (hasTransport) {
            if (line.hasOption("transportLayer") && transport == null) {
                String tmpTransport = line.getOptionValue("transportLayer");
                if (!BootstrapUtils.getBootstrapListOfTransports().contains(tmpTransport)) {
                    throw new IllegalArgumentException("Transport layer " + tmpTransport + " is not a valid value.");
                } else {
                    transport = tmpTransport;
                }
            }
        }

    }

    public static Properties getCmdLineProperties() {
        return bootstrapCmdLineProperties;
    }

    /**
     * The the name of the CCS application to be launched.
     *
     * @return The application name.
     */
    public static String getBootstrapApplication() {
//        if ( bootstrapApplication == null) {
//            throw new RuntimeException("The Bootstrap Application has not been defined yet.");
//        }
        return bootstrapApplication;
    }

    /**
     * Return true/false if help was requested.
     *
     * @return true/false if help was requested.
     */
    public boolean doPrintHelp() {
        return printHelp;
    }

    /**
     * Return true/false is verbose is turned on.
     *
     * @return true/false if verbose is turned on.
     */
    public boolean verbose() {
        return verbose;
    }

    /**
     * Get the bootstrap command line options.
     *
     * @return The Options containing the command line options.
     *
     */
    protected Options getBootstrapCommandLineOptions() {
        return bootstrapCommandLineOptions;
    }

    /**
     * Prints the content of a URLClassLoader.
     *
     * @param classLoader
     */
    private static void printBootstrapClassLoader() {
        System.out.println("*** CLASSPATH");
        URL[] urls = getBootstrapApplicationClassLoader().getURLs();
        for (URL url : urls) {
            System.out.println("\t\t"+url);
        }
    }

    /**
     * Get a given command line Option
     *
     * @param opt The option name.
     * @return The corresponding Option.
     */
    private Option getOption(String opt) {
        return getBootstrapCommandLineOptions().getOption(opt);
    }

    private void loadTransportPropertiesInSystemProperties(String transport) {
    }

    /**
     * Get the list of arguments to be passed to the application. This list
     * contains options that are transfered to the application (like -help and
     * -verbose), the list of arguments defined in the application property file
     * definition and all the arguments that are left after parsing.
     *
     * @param appProperties
     * @return
     */
    private String[] getApplicationArguments(Properties appProperties) throws ParseException {

        String applicationArgs = appProperties.getProperty(APPLICATION_ARGS_PROPERTY, "").trim();
        StringTokenizer applicationArgsTokenizer = new StringTokenizer(applicationArgs, " ");

        int nArgs = passedAlongOptions.size() + additionalCommandLineArguments.size() + applicationArgsTokenizer.countTokens();

        String[] mainArgs = new String[nArgs];

        int argCount = 0;
        while (applicationArgsTokenizer.hasMoreTokens()) {
            mainArgs[argCount++] = applicationArgsTokenizer.nextToken();
        }
        for (String opt : passedAlongOptions) {
            mainArgs[argCount++] = opt;
        }
        for (String cmdArg : additionalCommandLineArguments) {
            mainArgs[argCount++] = cmdArg;
        }
        if (verbose()) {
            System.out.print("*** Command line arguments passed to the mainClass: ");
            for (String arg : mainArgs) {
                System.out.print(arg + " ");
            }
            System.out.println();
        }

        return mainArgs;
    }

    /**
     * Get the classLoader for this application.
     *
     * @return The URLClassLoader.
     */
    public static URLClassLoader getBootstrapApplicationClassLoader() {
        if ( applicationClassLoader == null ) {
           buildBootstrapClassLoader();
        }
        return applicationClassLoader;
    }

    /**
     * Scan a given Properties file to search for properties that end with
     * "additional.classpath.entry". Do we enforce uniqueness? Do we load them
     * all?
     *
     * @param props
     */
    static void scanPropertiesForClassPathEntries(Properties props) {
        Set<Object> keySet = ResourcesUtils.getAllKeysInProperties(props);
        for (Object obj : keySet) {
            String key = (String) obj;
            if (key.endsWith("additional.classpath.entry")) {
                String entry = props.getProperty(key);
                if (additionalClassPathEntriesMap.containsKey(key)) {
                    System.out.println("*** [WARNING] ignoring additional classpath entry: " + key + "=" + entry + " it was already added to the CLASSPATH as " + additionalClassPathEntriesMap.get(key));
                } else {
                    additionalClassPathEntriesMap.put(key, entry);
                    additionalClassPathEntriesList.add(entry);
                }
            }
        }
    }

        
    private static void buildBootstrapClassLoader() {
        if ( bootstrapApplication == null ) {
            throw new RuntimeException("The Bootstrap Application has not been defined yet.");
        }
        
        if ( applicationClassLoader != null ) {
            throw new RuntimeException("The Bootstrap ClassLoader has already been built. Please report this problem.");
        }
        
        //Build the new classloader to launch the CCS application
        List<URL> classPathUrlList = new ArrayList<>();

        //Put all additional classpath entries from application definition file to classpath
        scanPropertiesForClassPathEntries(getBootstrapApplicationProperties());
        //TO-DO: Do we want to scan transport properties as well?


        try {

            //Add all the additional Entries to the classpath read from property files.
            for (String classPathEntry : additionalClassPathEntriesList) {
                File cpEntryFile = new File(classPathEntry);
                if ( cpEntryFile.isDirectory() ) {
                    System.out.println("*** [WARNING] we are currently ignoring directories as additional Classpath entries. Skipping "+classPathEntry);
                } else {
                    classPathUrlList.add(cpEntryFile.toURI().toURL());                    
                }                
            }


            //First add the mainJar
            String applicationMainJar = getApplicationMainJarFromProperties(getBootstrapApplicationProperties());
            classPathUrlList.add(new URL("file://" + applicationMainJar));
            //Then add what's defined in the manifest Class-Path element
            Manifest manifest = getBootsrapMainManifest();
            String manifestClassPath = manifest.getMainAttributes().getValue("Class-Path").trim();
            if (manifestClassPath != null) {
                StringTokenizer classPathTokens = new StringTokenizer(manifestClassPath, " ");
                while (classPathTokens.hasMoreTokens()) {
                    String manifestClassPathJar = classPathTokens.nextToken();
                    classPathUrlList.add(new URL("file://" + BootstrapUtils.getDistributionJarFilesDirectory() + manifestClassPathJar));
                }
            }


        } catch (MalformedURLException mue) {
            throw new RuntimeException("Failed to build URL when building the classpath: " + mue.getMessage());
        }

        //Build the array of classpath URLs
        URL[] classpathUrls = new URL[classPathUrlList.size()];
        int count = 0;
        for (URL classpathURl : classPathUrlList) {
            classpathUrls[count++] = classpathURl;
        }

        applicationClassLoader = new URLClassLoader(classpathUrls);
        Thread.currentThread().setContextClassLoader(applicationClassLoader);
        BootstrapClassLoader.addClassLoader(applicationClassLoader);
        
    }
    
    private static Manifest getBootsrapMainManifest() {
        //Get the main Jar for this application.
        String applicationMainJar = getApplicationMainJarFromProperties(getBootstrapApplicationProperties());
        Manifest manifest = null;
        try {
            JarFile applicationMainJarFile = new JarFile(applicationMainJar);
            manifest = applicationMainJarFile.getManifest();
        } catch (IOException ioe) {
            throw new RuntimeException("Could not extract Manifest from Jar file " + applicationMainJar, ioe);
        }
        return manifest;
    }
    
    /**
     * Get the fully qualified path to the application main jar file from the
     * application definition file.
     *
     * @param applicationName The name of the application.
     * @param applicationProperties The Properties file.
     * @return The main jar file path.
     */
    private static String getApplicationMainJarFromProperties(Properties applicationProperties) {
        String applicationMainJar = applicationProperties.getProperty(BootstrapUtils.APPLICATION_MAINJAR_PROPERTY);
        if (applicationMainJar == null) {
            throw new RuntimeException("*** Application Definition file must contain property " + BootstrapUtils.APPLICATION_MAINJAR_PROPERTY
                    + " in its definition file");
        }
        applicationMainJar = BootstrapUtils.getDistributionJarFilesDirectory() + applicationMainJar;
        return applicationMainJar;
    }
    
    
    /**
     * Launch the CCS Application for the given name.
     *
     * @param applicationName The name of the CCS application.
     */
    private void launchCCSApplication(String applicationName) {

        Properties applicationProperties = BootstrapUtils.getApplicationDefinitionProperties(applicationName);


        if (doPrintHelp()) {
            String appDescription = applicationProperties.getProperty(APPLICATION_DESCRIPTION_PROPERTY, "");
            System.out.println("\n\tCCS Application " + applicationName + " " + appDescription + "\n");
        }



        Manifest manifest = getBootsrapMainManifest();

        //Get the main Class to launch. Either from the properties or from the manifest.
        String manifestMainClass = manifest.getMainAttributes().getValue("Main-Class");
        String applicationMainClassName = applicationProperties.getProperty(APPLICATION_MAINCLASS_PROPERTY, manifestMainClass);

        if (applicationMainClassName == null) {
            throw new RuntimeException("*** Application " + applicationName + " must contain define the main Class to lauch. "
                    + "This can be done either in the Manifest of the main jar or by defining the property "
                    + APPLICATION_MAINCLASS_PROPERTY
                    + " in its definition file. ");
        }


        if (verbose()) {
            System.out.println("*** Distribution Root: " + BootstrapUtils.getCCSDistributionRootDirectory());
            System.out.println("*** Application name: " + applicationName);
            System.out.println("*** MainClass: " + applicationMainClassName);
            System.out.println("*** LD_LIBRARY_PATH: " + System.getenv("LD_LIBRARY_PATH"));
        }


        // The LD_LIBRARY_PATH environment variable is now set in the CCSbootstrap.sh script.
        //Set up the LD_LIBRARY_PATH
//        try {
//            //The following is a hack to make it work. 
//            //What follows is from http://blog.cedarsoft.com/2010/11/setting-java-library-path-programmatically/
//            //the only way to set the property java.libary.path is to add a system property *before* the application is started:
//            //
//            //java -Djava.library.path=/path/to/libs
//            //
//            //Changing the system property later doesn’t have any effect, since the property is evaluated very early and cached.
//            //But the Classloader has a static field (sys_paths) that contains the paths. 
//            //If that field is set to null, it is initialized automatically. 
//            //Therefore forcing that field to null will result into the reevaluation of the library path as soon as loadLibrary() is called…
//            System.setProperty("java.library.path", BootstrapUtils.getDistributionJniDirectory() + File.pathSeparator + BootstrapUtils.getDistributionLibDirectory());
//            Field fieldSysPath = ClassLoader.class.getDeclaredField("sys_paths");
//            fieldSysPath.setAccessible(true);
//            fieldSysPath.set(null, null);
//        } catch (NoSuchFieldException | SecurityException | IllegalArgumentException | IllegalAccessException e) {
//            System.out.println("*** Failed to set java.library.path");
//        }

        try {
            Class applicationMainClass = getBootstrapApplicationClassLoader().loadClass(applicationMainClassName);
            try {
                //Get the application main method.
                Method applicationMainMethod = applicationMainClass.getMethod("main",
                        new Class[]{String[].class});

                //Build the array containing the arguments to be passed to the main
                String[] mainArgs = getApplicationArguments(applicationProperties);

                //Now it is time to define the transport layer
                if (hasTransport) {
                    if (transport == null) {
                        transport = defaultTransport;
                    }
                    if (verbose()) {
                        System.out.println("*** Transport: " + transport);
                    }
                    loadTransportPropertiesInSystemProperties(transport);
                }

                //Print the classpath
                if (verbose()) {
                    printBootstrapClassLoader();
                }

                try {
                    //Invoke the main method on of the main Class.
                    applicationMainMethod.invoke(null, new Object[]{mainArgs});
                } catch (IllegalAccessException | IllegalArgumentException | InvocationTargetException e) {
                    System.out.println("*** Failed to invoke main method in class " + applicationMainClassName + "\n" + e.getMessage());
                    e.printStackTrace();
                }

            } catch (NoSuchMethodException | SecurityException | ParseException e) {
                System.out.println("*** Could not access the main method in class " + applicationMainClassName + "\n" + e.getMessage());
            }



        } catch (ClassNotFoundException cnfe) {
            System.out.println("*** Could not find class " + applicationMainClassName + " in the following classpath: ");
            printBootstrapClassLoader();
            System.out.println("*************************************************************************\n" + cnfe.getMessage());
        }

    }

    public static void main(String[] args) throws Exception {

        Bootstrap bootstrap = new Bootstrap();
        bootstrap.parseCommandLineArguments(args);

        String applicationName = getBootstrapApplication();

        if (bootstrap.showClasspath ) {
            if (getBootstrapApplication() == null) {
                throw new IllegalArgumentException("To display the content of the classpath you have to provide an application with the --app option");
            } else {
                printBootstrapClassLoader();
            }
        } else if (bootstrap.showProperties != null) {
            if (getBootstrapApplication() == null) {
                throw new IllegalArgumentException("To display the content of a properties file you have to provide an application with the --app option");
            } else {
                ResourcesUtils.printProperties(BootstrapUtils.getMergedProperties(bootstrap.showProperties));
            }
        } else if (bootstrap.showDistributionInfo) {
            printDistributionInfo();
        } else if (bootstrap.listApplications) {

            List<String> availableApplications = BootstrapUtils.getBootstrapListOfApplications();
            if (availableApplications.isEmpty()) {
                System.out.println("No CCS applications are defined in the current distribution.");
                if (bootstrap.verbose()) {
                    System.out.println(BootstrapUtils.getDistributionResourcesDirectory());
                }
            } else {
                System.out.println("Available CCS applications :");
                if (bootstrap.verbose()) {
                    System.out.println(BootstrapUtils.getDistributionResourcesDirectory());
                }
                for (String application : availableApplications) {
                    Properties applicationProps = BootstrapUtils.getApplicationDefinitionProperties(application);
                    System.out.println("\t" + application + "\t" + applicationProps.getProperty(APPLICATION_DESCRIPTION_PROPERTY));
                }
            }

        } else if (bootstrap.doPrintHelp() && applicationName == null) {
            printHelp(bootstrap.getBootstrapCommandLineOptions());
        } else if (applicationName != null) {
            bootstrap.launchCCSApplication(applicationName);
        } else {
            printHelp(bootstrap.getBootstrapCommandLineOptions());
        }

    }

    private static void printHelp(Options o) {
        HelpFormatter formatter = new HelpFormatter();
        formatter.printHelp(100, "CCSbootstrap", "", o, "", true);
    }

    private static void printDistributionInfo() {
        System.out.println("\n*** Distribution info");
        System.out.println("\tDistribution path: " + BootstrapUtils.getCCSDistributionRootDirectory());
        System.out.println("\tResources ordered search path: ");
        List<String> distSearchPathList = BootstrapUtils.getOrderedListOfResourceDirectories();
        String resourcesDirList = "";
        for (String dir : distSearchPathList) {
            resourcesDirList = resourcesDirList + "\t\t" + dir + "\n";
        }
        System.out.print(resourcesDirList);

    }
}
